<?php
/* --------------------------------------------------------------
   WithdrawalFactory.php 2020-01-23
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\Withdrawal;

use DateTime;
use Exception;
use Gambio\Admin\Withdrawal\Models\CustomerAddress;
use Gambio\Admin\Withdrawal\Models\CustomerDetails;
use Gambio\Admin\Withdrawal\Models\CustomerGender;
use Gambio\Admin\Withdrawal\Models\CustomerId;
use Gambio\Admin\Withdrawal\Models\OrderDetails;
use Gambio\Admin\Withdrawal\Models\OrderId;
use Gambio\Admin\Withdrawal\Models\Withdrawal;
use Gambio\Admin\Withdrawal\Models\WithdrawalId;
use InvalidArgumentException;

/**
 * Class WithdrawalFactory
 *
 * @package Gambio\Admin\Withdrawal
 */
class WithdrawalFactory
{
    /**
     * Creates a Withdrawal instance.
     *
     * @param int|null        $id
     * @param OrderDetails    $order
     * @param CustomerDetails $customer
     * @param string          $date
     * @param string          $content
     * @param bool            $createdByAdmin
     * @param string          $createdOn
     *
     * @return Withdrawal
     */
    public function createWithdrawal(
        OrderDetails $order,
        CustomerDetails $customer,
        string $date,
        string $content,
        bool $createdByAdmin,
        ?int $id = null,
        ?string $createdOn = null
    ): Withdrawal {
        try {
            $dateObj = new DateTime($date);
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for withdrawal date provided.');
        }
        try {
            $createdOnObj = ($createdOn !== null) ? new DateTime($createdOn) : null;
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for withdrawal creation date provided.');
        }
        
        return Withdrawal::create(WithdrawalId::create($id),
                                  $order,
                                  $customer,
                                  $dateObj,
                                  $content,
                                  $createdByAdmin,
                                  $createdOnObj);
    }
    
    
    /**
     * Create a OrderDetails instance.
     *
     * @param int|null    $id
     * @param string|null $creationDate
     * @param string|null $deliveryDate
     *
     * @return OrderDetails
     */
    public function createOrderDetails(
        ?int $id = null,
        ?string $creationDate = null,
        ?string $deliveryDate = null
    ): OrderDetails {
        try {
            $creationDateObj = ($creationDate !== null) ? new DateTime($creationDate) : $creationDate;
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for order creation date provided.');
        }
        try {
            $deliveryDateObj = ($deliveryDate !== null) ? new DateTime($deliveryDate) : $deliveryDate;
        } catch (Exception $e) {
            throw new InvalidArgumentException('Invalid datetime format for order delivery date provided.');
        }
        
        return OrderDetails::create(OrderId::create($id), $creationDateObj, $deliveryDateObj);
    }
    
    
    /**
     * Creates a CustomerDetails instance.
     *
     * @param int             $id
     * @param string          $gender
     * @param string          $firstname
     * @param string          $lastname
     * @param CustomerAddress $address
     * @param string          $email
     *
     * @return CustomerDetails
     */
    public function createCustomerDetails(
        string $gender,
        string $firstname,
        string $lastname,
        CustomerAddress $address,
        string $email,
        ?int $id = null
    ): CustomerDetails {
        return CustomerDetails::create(CustomerId::create($id),
                                       CustomerGender::create($gender),
                                       $firstname,
                                       $lastname,
                                       $address,
                                       $email);
    }
    
    
    /**
     * Creates a CustomerAddress instance.
     *
     * @param string $street
     * @param string $postcode
     * @param string $city
     * @param string $country
     *
     * @return CustomerAddress
     */
    public function createCustomerAddress(
        string $street,
        string $postcode,
        string $city,
        string $country
    ): CustomerAddress {
        return CustomerAddress::create($street, $postcode, $city, $country);
    }
}